//////////////////////////////////////////////
// main.cpp
//
//////////////////////////////////////////////

/// Includes ---------------------------------

// nkLog
#include <NilkinsLog/Loggers/ConsoleLogger.h>

// nkGraphics
#include <NilkinsGraphics/Cameras/Camera.h>
#include <NilkinsGraphics/Cameras/CameraManager.h>

#include <NilkinsGraphics/Encoders/Obj/ObjEncoder.h>

#include <NilkinsGraphics/Entities/Entity.h>

#include <NilkinsGraphics/Graph/Node.h>
#include <NilkinsGraphics/Graph/NodeManager.h>

#include <NilkinsGraphics/Log/LogManager.h>

#include <NilkinsGraphics/Meshes/Utils/MeshUtils.h>

#include <NilkinsGraphics/Meshes/Mesh.h>
#include <NilkinsGraphics/Meshes/MeshManager.h>

#include <NilkinsGraphics/Programs/Program.h>
#include <NilkinsGraphics/Programs/ProgramManager.h>
#include <NilkinsGraphics/Programs/ProgramSourcesHolder.h>

#include <NilkinsGraphics/RenderContexts/RenderContextDescriptor.h>
#include <NilkinsGraphics/RenderContexts/RenderContextManager.h>

#include <NilkinsGraphics/RenderQueues/RenderQueue.h>
#include <NilkinsGraphics/RenderQueues/RenderQueueManager.h>

#include <NilkinsGraphics/Samplers/Sampler.h>
#include <NilkinsGraphics/Samplers/SamplerManager.h>

#include <NilkinsGraphics/Shaders/Memory/ConstantBuffer.h>
#include <NilkinsGraphics/Shaders/Memory/ShaderInstanceMemorySlot.h>
#include <NilkinsGraphics/Shaders/Memory/ShaderPassMemorySlot.h>

#include <NilkinsGraphics/Shaders/Shader.h>
#include <NilkinsGraphics/Shaders/ShaderManager.h>

#include <NilkinsGraphics/System.h>

#include <NilkinsGraphics/Textures/Texture.h>
#include <NilkinsGraphics/Textures/TextureManager.h>

// nkResources
#include <NilkinsResources/ResourceManager.h>

// nkWinUi
#include <NilkinsWinUi/System.h>

// Standards
#include <chrono>

#include <memory>

/// Internals : Mesh -------------------------

void prepareMeshFromFile ()
{
	// Initialize mesh to use later
	nkGraphics::Mesh* mesh = nkGraphics::MeshManager::getInstance()->createOrRetrieve("Mesh") ;

	// Load data from file
	nkMemory::String absPath = nkResources::ResourceManager::getInstance()->getAbsoluteFromWorkingDir("sphere.obj") ;
	nkMemory::Buffer objData = nkResources::ResourceManager::getInstance()->loadFileIntoMemory(absPath) ;

	// We change some settings to alter the way the mesh is imported
	nkGraphics::ObjDecodeOptions objOptions ;
	objOptions._invertUvY = true ;
	objOptions._invertWindingOrder = true ;
	nkGraphics::DecodedData objDecoded = nkGraphics::ObjEncoder::decode(objData, objOptions) ;

	// Fill mesh
	nkGraphics::MeshFillOptions fillOptions ;
	fillOptions._autoLoad = true ;
	fillOptions._dataFillType = nkGraphics::DATA_FILL_TYPE::FORWARD ;
	nkGraphics::MeshUtils::fillMeshFromDecodedData(objDecoded._meshData[0], mesh, fillOptions) ;
}

void addMeshToRenderQueue ()
{
	nkGraphics::RenderQueue* rq = nkGraphics::RenderQueueManager::getInstance()->get(nkGraphics::RenderQueueManager::DEFAULT_RENDER_QUEUE) ;

	nkGraphics::Mesh* mesh = nkGraphics::MeshManager::getInstance()->createOrRetrieve("Mesh") ;

	nkGraphics::Entity* ent = rq->addEntity() ;
	ent->setRenderInfo(nkGraphics::EntityRenderInfo(mesh, nullptr)) ;
}

void addEntityToGraph ()
{
	nkGraphics::RenderQueue* rq = nkGraphics::RenderQueueManager::getInstance()->get(nkGraphics::RenderQueueManager::DEFAULT_RENDER_QUEUE) ;
	nkGraphics::Entity* ent = rq->getEntity(0) ;

	nkGraphics::Node* node = nkGraphics::NodeManager::getInstance()->createOrRetrieve("node") ;
	node->setPositionAbsolute(nkMaths::Vector(0.f, 0.f, 10.f)) ;

	ent->setParentNode(node) ;
}

/// Internals : Texture ----------------------

void prepareTexture ()
{
	// First bit to use a texture is to have a texture
	// We set its resource path, relative to Data, and ask it to load to seek the file and populate the GPU
	nkGraphics::Texture* tex = nkGraphics::TextureManager::getInstance()->createOrRetrieve("tex") ;

	tex->setPath("cloudMap.dds") ;
	tex->load() ;
}

void prepareSampler ()
{
	// Using textures also means we need a sampler to choose how to interpolate data sampled
	// We will use the default one, request it and load it right away
	nkGraphics::Sampler* sampler = nkGraphics::SamplerManager::getInstance()->createOrRetrieve("sampler") ;
	sampler->load() ;
}

/// Internals : Shader -----------------------

void prepareProgram ()
{
	// Program will remain straightforward, only addition is the use of UVs from the mesh and a texture and sampler
	nkGraphics::Program* program = nkGraphics::ProgramManager::getInstance()->createOrRetrieve("program") ;
	nkGraphics::ProgramSourcesHolder sources ;

	sources.setVertexMemory
	(
		R"eos(
			cbuffer PassBuffer : register(b0)
			{
				matrix view ;
				matrix proj ;
			}

			struct VertexInput
			{
				float4 position : POSITION ;
				float2 uvs : TEXCOORD0 ;
				matrix world : WORLD_MATRIX ;
			} ;

			struct PixelInput
			{
				float4 position : SV_POSITION ;
				float2 uvs : TEXCOORD0 ;
			} ;

			PixelInput main (VertexInput input)
			{
				PixelInput result ;

				matrix mvp = mul(proj, mul(view, input.world)) ;
				result.position = mul(mvp, input.position) ;
				result.uvs = input.uvs ;

				return result ;
			}
		)eos"
	) ;

	sources.setPixelMemory
	(
		R"eos(
			struct PixelInput
			{
				float4 position : SV_POSITION ;
				float2 uvs : TEXCOORD0 ;
			} ;

			Texture2D tex : register(t0) ;
			SamplerState customSampler : register(s0) ;
				
			float4 main (PixelInput input) : SV_TARGET
			{
				return tex.Sample(customSampler, input.uvs) ;
			}
		)eos"
	) ;

	program->setFromMemory(sources) ;
	program->load() ;
}

void prepareShader ()
{
	nkGraphics::Shader* shader = nkGraphics::ShaderManager::getInstance()->createOrRetrieve("shader") ;
	nkGraphics::Program* program = nkGraphics::ProgramManager::getInstance()->get("program") ;

	shader->setProgram(program) ;

	nkGraphics::ConstantBuffer* cBuffer = shader->addConstantBuffer(0) ;

	nkGraphics::ShaderPassMemorySlot* slot = cBuffer->addPassMemorySlot() ;
	slot->setAsViewMatrix() ;

	slot = cBuffer->addPassMemorySlot() ;
	slot->setAsProjectionMatrix() ;

	nkGraphics::ShaderInstanceMemorySlot* instanceSlot = shader->addInstanceMemorySlot() ;
	instanceSlot->setAsWorldMatrix() ;

	// Prepare for tex and sampler
	nkGraphics::Texture* tex = nkGraphics::TextureManager::getInstance()->get("tex") ;
	nkGraphics::Sampler* sampler = nkGraphics::SamplerManager::getInstance()->get("sampler") ;

	// Simply feed them through the shader, with their slots attached
	shader->addTexture(tex, 0) ;
	shader->addSampler(sampler, 0) ;

	shader->load() ;
}

void assignShader ()
{
	nkGraphics::RenderQueue* rq = nkGraphics::RenderQueueManager::getInstance()->get(nkGraphics::RenderQueueManager::DEFAULT_RENDER_QUEUE) ;
	nkGraphics::Entity* ent = rq->getEntity(0) ;
	nkGraphics::Shader* shader = nkGraphics::ShaderManager::getInstance()->get("shader") ;

	nkGraphics::Mesh* mesh = nkGraphics::MeshManager::getInstance()->createOrRetrieve("Mesh") ;
	ent->setRenderInfo(nkGraphics::EntityRenderInfo(mesh, shader)) ;
}

/// Function ---------------------------------

int main ()
{
	// Prepare for logging
	std::unique_ptr<nkLog::Logger> logger = std::make_unique<nkLog::ConsoleLogger>() ;
	nkGraphics::LogManager::getInstance()->setReceiver(logger.get()) ;

	// For easiness
	nkResources::ResourceManager::getInstance()->setWorkingPath("Data") ;

	// Initialize and create context with window
	if (!nkGraphics::System::getInstance()->initialize())
		return -1 ;

	nkGraphics::RenderContext* context = nkGraphics::RenderContextManager::getInstance()->createRenderContext(nkGraphics::RenderContextDescriptor(800, 600, false, true)) ;

	// Prepare the mesh we will show
	prepareMeshFromFile() ;
	addMeshToRenderQueue() ;
	addEntityToGraph() ;

	// Prepare the texture we will use
	prepareTexture() ;
	prepareSampler() ;

	// Change its shader
	prepareProgram() ;
	prepareShader() ;
	assignShader() ;

	// Get some data to prepare for the rendering loop
	nkGraphics::Camera* camera = nkGraphics::CameraManager::getInstance()->getDefaultCamera() ;

	nkGraphics::System* nkGraphicsSystem = nkGraphics::System::getInstance() ;
	nkWinUi::System* nkWinUiSystem = nkGraphicsSystem->getUiSystem() ;

	const float loopTimeMs = 5 * 1000.f ;
	std::chrono::system_clock::time_point firstNow = std::chrono::system_clock::now() ;

	const nkMaths::Vector sphereCenter = nkMaths::Vector(0.f, 0.f, 10.f) ;

	// Our rendering loop is ready to start
	while (nkGraphicsSystem->getHasRunToContinue())
	{
		// Frame the graphics system
		if (!nkGraphicsSystem->frame(context))
			break ;

		// Loop the camera around the sphere
		std::chrono::system_clock::time_point now = std::chrono::system_clock::now() ;
		float currentTimeMs = std::chrono::duration_cast<std::chrono::microseconds>(now - firstNow).count() / 1000.f ;
		
		float currentLoopT = (std::fmod(currentTimeMs, loopTimeMs) / loopTimeMs) * 2.f - 1.f ;
		currentLoopT *= 3.14 ;
		
		nkMaths::Vector loopPos (std::cos(currentLoopT) * 10.f, 0.f, std::sin(currentLoopT) * 10.f) ;
		loopPos += sphereCenter ;

		camera->setPositionAbsolute(loopPos) ;
		camera->lookAt(sphereCenter, nkMaths::Vector(0.f, 1.f, 0.f)) ;

		// Tick the windowing system
		nkWinUiSystem->tick() ;
	}

	// Clean exit
	nkGraphics::System::getInstance()->kill() ;
}